
;**********************************************************************
;   von_koch.pro    (Created by Rivix, LLC, 1998.)

;-----------------------------------
;   List of procedures in this file
;-----------------------------------
;   Plot_von_Koch_Fractal

;**********************************************************************
pro Plot_Von_Koch_Fractal, n, PS_file=PS_file, xsize=xsize, $
             x0=x0, y0=y0, length=length, angle=angle, $
             color=color, NOT_ROOT=NOT_ROOT, level=level 

;---------------------------------------------------------
;NOTES:  This routine recursively draws the famous von
;        Koch fractal curve.  It is a good example of how
;        to draw fractals with IDL.  See SSTs.pro for
;        another example.

;        Normal coordinates are used to make the routine
;        device-independent (window or PS file).

;        Do not use the NOT_ROOT or "level" keywords!
;        They are used by the program to track the
;        level of recursion.

;        Set the PS_file keyword to a filename path in
;        single quotes with extension '.ps' in order to
;        save the image to a PostScript file.

;        The "xsize" keyword sets xsize of plot window.
;        A large value like 900 or 1000 helps to see
;        fine detail for n >= 5.  

;        n      = number of levels to draw
;        x0     = x-coordinate of start 
;        y0     = y-coordinate of start
;        angle  = angle with horizontal 
;        length = length scale for level 1
;----------------------------------------------------------

;-----------------
;Keyword defaults
;-----------------
S = sqrt(3.0)/6.0
if (n_elements(n) eq 0) then n=4
n = ((n > 1) < 8)
if NOT(keyword_set(x0)) then x0 = 0.05  ;(normal coords)
if NOT(keyword_set(y0)) then y0 = 0.05  ;(normal coords)
if NOT(keyword_set(angle)) then angle = 0d
if NOT(keyword_set(length)) then length = (1.0 - (2*x0))
if NOT(keyword_set(xsize)) then xsize = 900

NOT_ROOT = keyword_set(NOT_ROOT)

;---------------------------
;Only called the first time
;---------------------------
if NOT(keyword_set(level)) then begin
    level = n
    ;-----------------------------------
    ;Load color table "Rainbow + white"
    ;-----------------------------------
    loadct, 39, /silent
    black = 0
    white = (!d.table_size - 1)
    color = black
    if (keyword_set(PS_File)) then begin
        ;-------------------------------
        ;Will plot to a PostScript file
        ;-------------------------------
        win_system = !d.name
        set_plot, 'PS'
        device, file=PS_file
    endif else begin
        ;-------------------------------
        ;Will plot to a graphics window
        ;-------------------------------
        device, decomposed=0

        window, 0, xsize=xsize, ysize=(S * xsize), $
                xpos=50, TITLE='The von Koch Fractal Curve'
        erase, white
    endelse
    print,'Plotting the von Koch fractal curve...'
endif

ang = fltarr(4)
x   = fltarr(5)
y   = fltarr(5)
L   = (length / 3.0)

;---------------------------------
;Note double-precision for angles
;---------------------------------
ang  = angle  + [0d, (!PI/3d), (-1*!PI/3d), 0d]

x[0] = x0 
x[1] = x[0] + (L * cos(ang[0]))
x[2] = x[1] + (L * cos(ang[1]))
x[3] = x[2] + (L * cos(ang[2])) 
x[4] = x[3] + (L * cos(ang[3])) 
y[0] = y0 
y[1] = y[0] + (L * sin(ang[0]) / S)
y[2] = y[1] + (L * sin(ang[1]) / S)
y[3] = y[2] + (L * sin(ang[2]) / S) 
y[4] = y[3] + (L * sin(ang[3]) / S) 

if (level gt 1) then begin
    ;-----------------------------------
    ;Recursive call to draw first piece
    ;-----------------------------------
    Plot_Von_Koch_Fractal, level=(level-1), x0=x[0], y0=y[0], $
             len=L, angle=ang[0], color=color, /NOT_ROOT

    ;------------------------------------
    ;Recursive call to draw second piece
    ;------------------------------------
    Plot_Von_Koch_Fractal, level=(level-1), x0=x[1], y0=y[1], $
             len=L, angle=ang[1], color=color, /NOT_ROOT

    ;-----------------------------------
    ;Recursive call to draw third piece
    ;-----------------------------------
    Plot_Von_Koch_Fractal, level=(level-1), x0=x[2], y0=y[2], $
             len=L, angle=ang[2], color=color, /NOT_ROOT

    ;------------------------------------
    ;Recursive call to draw fourth piece
    ;------------------------------------
    Plot_Von_Koch_Fractal, level=(level-1), x0=x[3], y0=y[3], $
             len=L, angle=ang[3], color=color, /NOT_ROOT

endif else begin
    plots,[x[0],x[1]],[y[0],y[1]], color=color, /normal
    plots,[x[1],x[2]],[y[1],y[2]], color=color, /normal
    plots,[x[2],x[3]],[y[2],y[3]], color=color, /normal
    plots,[x[3],x[4]],[y[3],y[4]], color=color, /normal
endelse

if NOT(NOT_ROOT) then begin
   ;-----------------------------------
   ;Option to draw bounding box around
   ;the curve for PostScript file.
   ;-----------------------------------
   ;plots,[0,0],[1,0], /normal
   ;plots,[1,0],[1,1], /normal
   ;plots,[1,1],[0,1], /normal
   ;plots,[0,1],[0,0], /normal

   ;-----------------------------
   ;Toggle back to window system
   ;-----------------------------
   if (keyword_set(PS_file)) then begin
       device, /close
       set_plot, win_system
   endif
   print,'Finished.'
endif
 
end;  Plot_Von_Koch_Fractal
;*****************************************************************
