
;*****************************************************************
;   scheidegger.pro   (Created by Rivix, LLC, 1998.)

;---------------------------------
;   List of routines in this file
;---------------------------------
;   New_String
;   Plot_Scheidegger

;*****************************************************************
function New_String, number

RETURN, strtrim(string(number), 2)
end;  New_String
;*****************************************************************
pro Plot_Scheidegger, numy=numy, xsize=xsize, ysize=ysize, $
                      prob=prob, REPORT=REPORT, PS_file=PS_File

;-------------------------------------------------------------
;NOTES:  The Scheidegger model is a theoretical model for the
;        formation of channels in steep, mountain catchments.
;        It has been studied in connection with Hack's law.
;        The random tree is drawn on a triangular lattice,
;        which is built out of equilateral triangles.  Notice
;        that dy = (dx * sqrt(3)/2).

;        The xsize and ysize keywords set window size.

;        Normal coordinates are used to make the routine
;        device-independent (window or PS file).

;        Set the PS_file keyword to a filename path in
;        single quotes with extension '.ps' in order to
;        save the image to a PostScript file.

;        prob = probability of drawing segment down to the
;               right; and (1-prob) down to the left
;------------------------------------------------------------

;-----------------
;Keyword defaults
;-----------------
REPORT = keyword_set(REPORT)
if NOT(keyword_set(prob))  then prob  = 0.5
if NOT(keyword_set(numy))  then numy  = 50
if NOT(keyword_set(xsize)) then xsize = 500
if NOT(keyword_set(ysize)) then ysize = 400

;-----------------------------------
;Load color table "Rainbow + white"
;-----------------------------------
loadct, 39, /silent
black = 0
white = (!d.table_size - 1)
color = black
if (keyword_set(PS_File)) then begin
    ;-------------------------------
    ;Will plot to a PostScript file
    ;-------------------------------
    win_system = !d.name
    set_plot, 'PS'
    device, file=PS_file
endif else begin
    ;-------------------------------
    ;Will plot to a graphics window
    ;-------------------------------
    device, decomposed=0
    window, 0, xsize=xsize, ysize=ysize, $
            xpos=100, TITLE='Scheidegger Model Network'
    erase, white
endelse

;----------------------------------------
;Assume xsize, ysize and numy are given.
;Formulas below may seem strange, but
;this is because we must have xmax=ymax=
;1.0 regardless of aspect ratio, AR.
;---------------------------------------
AR   = float(xsize) / float(ysize)
ymax = 1d
xmax = 1d
dy   = (ymax / numy)
dx   = ((2d / sqrt(3d)) * dy) / AR
numx = fix(xmax / dx)
;---------------------------------
x1   = (dx * indgen(numx + 1))
x2   = (x1[0:numx-1] + (dx / 2d))
y    = ymax

print,'Plotting Scheidegger model network...'

;----------------
;Create the plot
;----------------
for j=0,(numy - 1) do begin
    for i=1,(numx - 1) do begin
        ;---------------------------
        ;Get random number in [0,1]
        ;---------------------------
        u = randomu(seed)

        if ((j mod 2) eq 0) then begin
            ;---------------------------
            ;This is an even row number
            ;---------------------------
            if (u lt prob) then begin
                plots,[x1[i],x2[i]],  [y,y - dy], $
                      color=color, /normal
            endif else begin
                plots,[x1[i],x2[i-1]],[y,y - dy], $
                      color=color, /normal
            endelse
        endif else begin
            ;--------------------------
            ;This is an odd row number
            ;--------------------------
            if (u gt prob) then begin
                 plots, [x2[i],x1[i]],  [y,y - dy], $
                        color=color, /normal
            endif else begin
                 plots, [x2[i],x1[i+1]], [y,y - dy], $
                        color=color, /normal
            endelse
        endelse
    endfor
    y = (y - dy)
endfor

;-----------------------------
;Toggle back to window system
;-----------------------------
if (keyword_set(PS_file)) then begin
    device, /close
    set_plot, win_system
endif
print,'Finished.'

;----------------
;Optional report
;----------------
if (REPORT) then begin
    print,'xsize = ' + New_String(xsize)
    print,'ysize = ' + New_String(ysize)
    print,'xmax  = ' + New_String(xmax)
    print,'ymax  = ' + New_String(ymax)
    print,'numx  = ' + New_String(numx)
    print,'numy  = ' + New_String(numy)
    print,'dx    = ' + New_String(dx)
    print,'dy    = ' + New_String(dy)
    print,'prob  = ' + New_String(prob)
    print,' '
endif

end;  Plot_Scheidegger
;*****************************************************************
