
;*****************************************************************
;    resized_grid.pro  (Rivix, LLC. 1998)

;    Resized_Grid

;*****************************************************************
function Resized_Grid, grid, factor, REDUCE, SAMPTYPE, $
                       new_ncols, new_nrows

;--------------------------------------------------------------
;NOTES:  This routine allows any 2D grid to be "rebinned"
;        to n times larger or smaller than its original size,
;        where n is a positive integer called "factor".

;        If (REDUCE eq 1b) then the grid is reduced in size
;        by sampling at regular, equal-spaced intervals.

;        If (REDUCE eq 0b) then the grid is enlarged in size
;        by pixel replication.

;        This routine reduces to IDL's REBIN routine if:
;           (1) (REDUCE eq 0), or
;           (2) (REDUCE eq 1) AND the requested grid
;                dimensions are multiples of the original
;                dimensions.
;        When these conditions are not met, the original
;        grid is cropped by the smallest amount necessary
;        to meet them.  This amount will never exceed n.

;        Note that IDL's REBIN routine only allows rebinning
;        if new grid dimensions are multiples or factors of
;        the old grid dimensions, which is often not the case
;        when a grid is reduced in size.

;        Resizing by arbitrary scale factors is avoided
;        because it distorts the data too much.
;--------------------------------------------------------------

if (factor le 0) then begin
    print,'Scale factor must be greater than zero.'
    print,' '
    RETURN,0
endif

if (n_elements(SAMPTYPE) eq 0) then SAMPLE_TYPE=1

s     = size(grid, /dimensions)
ncols = s[0]
nrows = s[1]

if (factor ne 1) then begin
    if (REDUCE) then begin
        new_ncols = long(ncols / factor) > 1L
        new_nrows = long(nrows / factor) > 1L

        x_rem = (ncols mod new_ncols)
        y_rem = (nrows mod new_nrows)

        if ((x_rem eq 0) AND (y_rem eq 0)) then begin
            a = rebin(grid, new_ncols, new_nrows, sample=SAMPLE_TYPE)
        endif else begin
            a = rebin(grid[0:(ncols - 1L - x_rem), $
                           0:(nrows - 1L - y_rem)], $
                       new_ncols, new_nrows, sample=SAMPLE_TYPE)
        endelse
    endif else begin
        new_ncols = long(factor) * ncols
        new_nrows = long(factor) * nrows
        a = rebin(grid, new_ncols, new_nrows, sample=SAMPLE_TYPE)
    endelse
endif else begin
    new_ncols = ncols
    new_nrows = nrows
    RETURN, grid
endelse

RETURN, a
end;  Resized_Grid
;*****************************************************************
