
;******************************************************************
;   fractal_surf.pro.   Created by S. Peckham, July 2004

;   The Make_Fractal_Surface routine uses the random midpoint
;   displacement method for creating a fractal surface/landscape.
;   The Make_Noise_Surface routine creates a grid with white
;   (uncorrelated) noise.  The remaining routines are utilities
;   that allow the resulting grids to be saved as RiverTools
;   Grid (RTG) files.

;-------------------------------
;   IDL procedures in this file
;-------------------------------
;   New_String
;   Get_Byte_Order 
;   Get_RTI_Record
;   Make_RTI_File

;   Make_Fractal_Surface
;   Make_Noise_Surface

;*****************************************************************
function New_String, number 

;--------------------
;Check for byte type
;--------------------
s = size(number)
if (s[1] eq 1) then num=fix(number) else num=number

RETURN, strtrim(string(num), 2)
END;  New_String
;***************************************************************
pro Get_Byte_Order, machine_byte_order

big_endian = (byte(1,0,2))[0] eq 0b
if (big_endian) then machine_byte_order='MSB' $
                else machine_byte_order='LSB'

END;  Get_Byte_Order
;***************************************************************
pro Get_RTI_Record, info

;------------------------------------------------------
;NOTES:  Shouldn't add INFO_FILE or DEM_FILE keywords.
;------------------------------------------------------

NODATA  = -9999
NODATAI = fix(NODATA)
NODATAL = long(NODATA)
NODATAD = double(NODATA)

;----------------------------------------
;Default is byte order of this machine
;----------------------------------------
Get_Byte_Order, byte_order

info = { $
DEM_file:    'Unknown', $
RTI_file:    'Unknown', $
data_source: 'Unknown', $
ncols:       NODATAL, $
nrows:       NODATAL, $
data_type:   'UNKNOWN_TYPE', $
byte_order:  byte_order, $
;---------------------------------
pixel_geom:0b, $
xres: NODATAD, yres: NODATAD, zres:1d, $
z_units: 'METERS', $
;---------------------------------
y_north_edge: NODATAD, $
y_south_edge: NODATAD, $
x_west_edge:  NODATAD, $
x_east_edge:  NODATAD, $
;---------------------------------
box_units: 'DEGREES', $
emin:NODATA, emax:NODATA, $
UTM_zone:'unknown'}

END;  Get_RTI_Record
;*****************************************************************
pro Make_RTI_File, RTI_file, info, SILENT=SILENT

;-------------------------------------------------------------
;NOTES:  type is in {'INTEGER','FLOAT','LONG','DOUBLE'}.

;        info.DEM_file should not be a full file
;        path because path info will change if file is moved.
;        If it is, for any reason, it is now shortened to
;        just a filename before writing to RTI file.
;        Note that if a comma is used instead of a plus sign
;        in the line that writes DEM_file, then the line
;        could span two lines and cause errors on read.
;-------------------------------------------------------------

SILENT = keyword_set(SILENT)
NODATA = -9999
f = '(D22.12)'
y_south_string = strtrim(string(info.y_south_edge, format = f), 2)
y_north_string = strtrim(string(info.y_north_edge, format = f), 2)
x_east_string  = strtrim(string(info.x_east_edge,  format = f), 2)
x_west_string  = strtrim(string(info.x_west_edge,  format = f), 2)

;-------------------------------------
;Get DEM name from possible filepath
;See NOTES at top.
;-------------------------------------
DEM_name = info.DEM_file

;-----------------------
;Open RTI file to write
;-----------------------
openw, A, RTI_file, /get_lun

;-----------------------
;Write info to RTI file
;-----------------------
printf,A,'RiverTools Info File'
printf,A,' '
printf,A,';--------------------'
printf,A,';Description of Grid'
printf,A,';--------------------'
printf,A,'DEM filename:  ' + DEM_name
printf,A,'DEM source:    ' + info.data_source
printf,A,' '
printf,A,';----------------------'
printf,A,';Grid dimensions, etc.'
printf,A,';----------------------'
printf,A,'Number of columns:  ' + New_String(info.ncols)
printf,A,'Number of rows:     ' + New_String(info.nrows)
printf,A,'Data type:          ' + info.data_type
printf,A,'Byte order:         ' + info.byte_order
printf,A,' '
printf,A,';--------------------'
printf,A,';Pixel geometry info'
printf,A,';--------------------'
printf,A,'Pixel geometry code:  ' + New_String(fix(info.pixel_geom))
printf,A,'Pixel x-resolution:   ' + New_String(info.xres)
printf,A,'Pixel y-resolution:   ' + New_String(info.yres)
printf,A,'Pixel z-resolution:   ' + New_String(info.zres)
printf,A,'Z-resolution units:   ' + info.z_units
printf,A,' '
printf,A,';--------------------------------------------------'
printf,A,';Bounding box coordinates (degrees lat/lon or UTM)'
printf,A,';--------------------------------------------------'
printf,A,'South edge y-coordinate:  ' + y_south_string
printf,A,'North edge y-coordinate:  ' + y_north_string
printf,A,'East  edge x-coordinate:  ' + x_east_string
printf,A,'West  edge x-coordinate:  ' + x_west_string
printf,A,'Measurement units:        ' + info.box_units
printf,A,' '
printf,A,';----------------------'
printf,A,';Min and max elevation'
printf,A,';----------------------'
printf,A,'Minimum elevation:  ' + New_String(info.emin)
printf,A,'Maximum elevation:  ' + New_String(info.emax)
printf,A,' '
printf,A,';---------'
printf,A,';UTM zone'
printf,A,';---------'
printf,A,'UTM zone: ' + info.UTM_zone
printf,A,' '
printf,A,' '

;-----------------------
;Write RTI file trailer
;-----------------------
printf,A,';----------------------------------'
printf,A,';Notes About RiverTools Info Files'
printf,A,';----------------------------------'
printf,A,";(1)  RiverTools grid info filenames end with '.rti'."
printf,A,';(2)  The first line should be:  RiverTools Info File'
printf,A,';(3)  Lines starting with a semi-colon are ignored.'
printf,A,';(4)  Colons are used to delimit labels from values.'
printf,A,';(5)  The order of the numbers above is important.'
printf,A,';(6)  Number of rows and columns are required.'
printf,A,';(7)  Pixel geometry codes are: 0=Fixed-Angle, ' + $
         '1=Fixed-Length.'
printf,A,';(8)  Pixel x-resolution and y-resolution are required.'
printf,A,';(9)  Measurement units must be METERS or DEGREES.'
printf,A,';(10) Elevation data type is required.'
printf,A,';     Allowed types are BYTE, INTEGER, LONG, FLOAT, DOUBLE.'
printf,A,';(11) Byte order is required; either LSB or MSB.'
printf,A,';     (LSB = Least Significant Byte = little-endian)'
printf,A,';     (MSB = Most Significant Byte  = big-endian)'
printf,A,";(12) For 'fixed-angle' pixels, bounding lats and lons"
printf,A,';     are required to compute lengths and areas correctly.'
printf,A,';(13) Latitudes south of equator are negative and'
printf,A,';     longitudes west of prime meridian are negative.'
printf,A,';(14) This file is best modified with the View DEM Info'
printf,A,';     dialog in the File menu.'
printf,A,' '
printf,A,' '

;-----------------------------
;Close file and print message
;-----------------------------
free_lun, A

if NOT(SILENT) then begin
    print,'DEM info written to: '
    print, RTI_file
endif

END;  Make_RTI_File
;******************************************************************
pro Make_Fractal_Surface, n_levels, H, g, RTG_FILE=RTG_file, $
                 SIGMA=SIGMA, SCALE=SCALE, X_WRAP=X_WRAP, $
                 Y_WRAP=Y_WRAP

;------------------------------------------------------------
;Notes:  n_levels is the number of fractal levels and
;        determines the dimensions of the resulting grid, g.

;        If n_levels = 7,  nx = 129
;        If n_levels = 8,  nx = 257
;        If n_levels = 9,  nx = 513
;        If n_levels = 10, nx = 1025
;        If n_levels = 11, nx = 2049

;        H is the Hurst exponent, and should be between
;        0 and 1.  The fractal dimension of the resulting
;        fractal surface will be D = (3-H), and will be
;        between 2 and 3.  Try H around 0.8 to 0.9 so that
;        D is between 2.1 and 2.2.  You can also experiment
;        with different values of sigma by setting the
;        SIGMA keyword.

;        Set the RTG_FILE keyword to the name of a file
;        with extension '.rtg' in single quotes in order to
;        save the resulting grid as a RiverTools Grid file.

;        Set the X_WRAP or Y_WRAP keywords in order to
;        impose periodic boundaries on the left & right
;        or top & bottom, respectively.

;        Set the SCALE keyword to rescale at very end.
;        A value around 150 works fairly well.

;---------------------------------------------------------
;*** seed = 168993

if (n_levels gt 11) then begin
    print,'********************************************'
    print,'ERROR: Max number of levels is 11,'
    print,'which gives a grid size of 2049 x 2049.'
    print,'********************************************'
    print,' '
    RETURN
endif

X_WRAP = keyword_set(X_WRAP)
Y_WRAP = keyword_set(Y_WRAP)

if NOT(keyword_set(SIGMA)) then SIGMA = 1d

print,'Creating fractal surface...'

;----------------
;Initialize vars
;----------------
nx  = (2L^n_levels) + 1
ny  = nx
step = nx - 1
ones = intarr(ny) + 1
cols = indgen(nx) # ones
rows = ones # indgen(ny)
sums = (cols + rows)
DONE = bytarr(nx, ny)
;----------------------
EDGE = bytarr(nx, ny)
EDGE[0,*]    = 1b
EDGE[nx-1,*] = 1b
EDGE[*,0]    = 1b
EDGE[*,ny-1] = 1b

;----------------
;Initialize grid
;----------------
g = dblarr(nx, ny)
v = randomn(seed, 2, 2, /double)
g[0,0]       = v[0,0]
g[nx-1,0]    = v[1,0]
g[0,ny-1]    = v[0,1]
g[nx-1,ny-1] = v[1,1]
;----------------------------
if (X_WRAP) then begin
    g[nx-1,0]    = g[0,0]
    g[nx-1,ny-1] = g[0,ny-1]
endif
if (Y_WRAP) then begin
    g[0,ny-1]    = g[0,0]
    g[nx-1,ny-1] = g[nx-1,0]
endif
;----------------------------
DONE[0,0]       = 1b
DONE[nx-1,0]    = 1b
DONE[0,ny-1]    = 1b
DONE[nx-1,ny-1] = 1b
;----------------------------
factor = 1d / sqrt(2d^H)  ;*********

for k=1, n_levels do begin

    step = (step / 2)

    ;-------------------------------------
    ;Get midpoint locations of this level
    ;-------------------------------------
    w = where(((cols mod step) eq 0) AND $
              ((rows mod step) eq 0) AND $
                NOT(DONE) and NOT(EDGE), n_mid)
   
    ;---------------------------------------
    ;Break these into two groups, w1 and w2
    ;---------------------------------------
    a1 = where((sums[w] mod (2*step)) eq 0, n1, COMP=a2, NCOMP=n2)
    if (n1 ne 0) then w1 = w[a1]
    if (n2 ne 0) then w2 = w[a2]

    ;-------------------------------------------
    ;Compute midpoint elevations as the average
    ;of the diagonal neighbor elevations plus
    ;a rescaled Gaussian random variable
    ;-------------------------------------------
    UL = w1 - step*(nx+1)
    UR = w1 - step*(nx-1)
    LL = w1 + step*(nx-1)
    LR = w1 + step*(nx+1)
    ;---------------------------
    ran   = factor * sigma * randomn(seed, n1, /double)
    g[w1] = ((g[UL] + g[UR] + g[LL] + g[LR])/4d) + ran
    DONE[w1] = 1b

    ;--------------------------------------------
    ;Compute midpoint elevations of remaining
    ;pixels at this scale as the average of the
    ;nearest neighbor elevations plus a rescaled
    ;Gaussian random variable.  n2=0 at start.
    ;--------------------------------------------
    if (n2 ne 0) then begin
        T = w2 - (step * nx) 
        B = w2 + (step * nx)
        R = w2 + step
        L = w2 - step
        ;---------------------
        ran   = factor * sigma * randomn(seed, n2, /double)
        g[w2] = ((g[T] + g[B] + g[L] + g[R])/4d) + ran
        DONE[w2] = 1b
    endif

    ;------------------------------------------
    ;Compute elevations of edge pixels at this
    ;scale as average of 3 nearest neighbors
    ;plus a rescaled Gaussian random variable.
    ;------------------------------------------
    jump = (step * nx)
    ;----------------------------
    L = where((cols eq 0) AND $
             ((rows mod step) eq 0) AND NOT(DONE), nL)
    T = L - jump
    B = L + jump
    R = L + step
    ran = factor * sigma * randomn(seed, nL, /double)
    g[L] = ((g[T] + g[B] + g[R])/3d) + ran
    DONE[L] = 1b
    ;--------------------------------------------------
    R = where((cols eq (nx-1)) AND $
             ((rows mod step) eq 0) AND NOT(DONE), nR)
    if NOT(X_WRAP) then begin
        L = R - step
        T = R - jump
        B = R + jump
        ran = factor * sigma * randomn(seed, nR, /double)
        g[R] = ((g[L] + g[T] + g[B])/3d) + ran
    endif else begin
        g[R] = g[L]
    endelse
    DONE[R] = 1b
    ;--------------------------------------------------
    T = where((rows eq 0) AND $
             ((cols mod step) eq 0) AND NOT(DONE), nT)
    L = T - step
    R = T + step
    B = T + jump
    ran = factor * sigma * randomn(seed, nT, /double)
    g[T] = ((g[L] + g[R] + g[B])/3d) + ran
    DONE[T] = 1b
    ;--------------------------------------------------
    B = where((rows eq (ny-1)) AND $
             ((cols mod step) eq 0) AND NOT(DONE), nB)
    if NOT(Y_WRAP) then begin
        L = B - step
        R = B + step
        T = B - jump
        ran = factor * sigma * randomn(seed, nB, /double)
        g[B] = ((g[L] + g[R] + g[T])/3d) + ran
    endif else begin
        g[B] = g[T]
    endelse
    DONE[B] = 1b
    ;--------------------------------------------------
 endfor 

;---------------------
;Rescale the values ?
;---------------------
if (keyword_set(SCALE)) then begin
    g = g * SCALE
endif

;---------------------------
;Option to save to RTG file
;---------------------------
if (keyword_set(RTG_FILE)) then begin
    openw, unit, RTG_file, /get_lun
    writeu, unit, g
    free_lun, unit

    ;-----------------
    ;Make an RTI file
    ;-----------------
    Get_RTI_Record, info
    Get_Byte_Order, byte_order
    emin = min(g, max=emax)
    xres = 100.0
    yres = 100.0
    ;-------------------------------
    p = strpos(RTG_file, '.')
    prefix = strmid(RTG_file, 0, p)
    len    = strlen(prefix)
    last4  = strupcase(strmid(prefix, len-4, 4))
    if (last4 eq '_DEM') then prefix=strmid(prefix,0,len-4)
    RTI_file = (prefix + '.rti')
    ;-------------------------------
    info.DEM_file     = RTG_file
    info.RTI_file     = RTI_file
    info.ncols        = nx
    info.nrows        = ny
    info.xres         = xres
    info.yres         = yres 
    info.zres         = 1.0
    info.data_type    = 'DOUBLE'
    info.byte_order   = byte_order
    info.data_source  = "Midpoint displacement method"
    info.pixel_geom   = 1b
    info.z_units      = 'METERS'
    info.y_south_edge = 0.0
    info.y_north_edge = (yres * ny)
    info.x_west_edge  = 0.0
    info.x_east_edge  = (xres * nx)
    info.box_units    = 'METERS'
    info.emin         = emin
    info.emax         = emax
    ;-------------------------------
    Make_RTI_File, RTI_file, info
endif

print,'Finished.'
print,' '

end;  Make_Fractal_Surface
;******************************************************************
pro Make_Noise_Surface, nx, g, RTG_FILE=RTG_file, SCALE=SCALE, $
                        SIGMA=sigma

print,'Creating white noise surface...'

if NOT(keyword_set(SIGMA)) then sigma = 1d

ny = nx
g  = sigma * randomn(seed, nx, ny, /double)

;---------------------
;Rescale the values ?
;---------------------
if (keyword_set(SCALE)) then begin
    g = g * SCALE
endif

;---------------------------
;Option to save to RTG file
;---------------------------
if (keyword_set(RTG_FILE)) then begin
    openw, unit, RTG_file, /get_lun
    writeu, unit, g
    free_lun, unit

    ;-----------------
    ;Make an RTI file
    ;-----------------
    Get_RTI_Record, info
    Get_Byte_Order, byte_order
    emin = min(g, max=emax)
    xres = 100.0
    yres = 100.0
    ;-------------------------------
    p = strpos(RTG_file, '.')
    prefix = strmid(RTG_file, 0, p)
    len    = strlen(prefix)
    last4  = strupcase(strmid(prefix, len-4, 4))
    if (last4 eq '_DEM') then prefix=strmid(prefix,0,len-4)
    RTI_file = (prefix + '.rti')
    ;-------------------------------
    info.DEM_file     = RTG_file
    info.RTI_file     = RTI_file
    info.ncols        = nx
    info.nrows        = ny
    info.xres         = xres
    info.yres         = yres 
    info.zres         = 1.0
    info.data_type    = 'DOUBLE'
    info.byte_order   = byte_order
    info.data_source  = "Midpoint displacement method"
    info.pixel_geom   = 1b
    info.z_units      = 'METERS'
    info.y_south_edge = 0.0
    info.y_north_edge = (yres * ny)
    info.x_west_edge  = 0.0
    info.x_east_edge  = (xres * nx)
    info.box_units    = 'METERS'
    info.emin         = emin
    info.emax         = emax
    ;-------------------------------
    Make_RTI_File, RTI_file, info
endif

print,'Finished.'
print,' '

end;  Make_Noise_Surface
;******************************************************************
