
;**********************************************************************
;   bin_cascade.pro   (Created by Rivix, LLC, 1998.)

;-----------------------------------
;   List of procedures in this file
;-----------------------------------
;   New_String
;   Plot_Binomial_Cascade

;*****************************************************************
function New_String, number, FORMAT=FORMAT

RETURN, strtrim(string(number, FORMAT=FORMAT), 2)
end;  New_String
;**********************************************************************
pro Plot_Binomial_Cascade, p, nlevels, PS_file=PS_file, $
                  list=list, LENGTH=LENGTH, $
                  REPORT=REPORT, NO_PLOT=NO_PLOT, $
                  margin=margin, XSIZE=xsize, YSIZE=ysize

;-------------------------------------------------------- 
;NOTE:  This routine recursively generates the sequence
;       of numbers in a binomial cascade with parameter
;       p, at level n, and plots them as a bar plot.

;       PostScript may not work correctly.

;       Don't use p and nlevels as keywords.
;--------------------------------------------------------

;------------------
;Argument defaults
;------------------
if (n_elements(p) eq 0)     then p = 0.3
if (n_elements(nlevels) eq 0) then nlevels = 6
if NOT(keyword_set(margin)) then margin = 0.05d
if NOT(keyword_set(XSIZE)) then xsize=700
if NOT(keyword_set(YSIZE)) then ysize=400
xplot = (1d - (2d * margin))
yplot = (1d - (2d * margin))
 
;------------------------------------
;Compute the binomial cascade values
;------------------------------------
p = double(p) 
list = [p, (1-p)]
for k=1L,(nlevels - 1L) do list=[p*list,(1-p)*list]
n = n_elements(list)
LENGTH = n      ;(to be returned)

;----------------
;Optional report
;----------------
if keyword_set(REPORT) then begin
   LMIN = min(list, MAX=LMAX)
   print,'------------------------------------'
   print,'Parameter, p    = ' + New_String(p)
   print,'No. of levels   = ' + New_String(nlevels)
   print,'No. of elements = ' + New_String(n)
   print,'Minimum value   = ' + New_String(LMIN)
   print,'Maximum value   = ' + New_String(LMAX)
   print,'------------------------------------'
   print,' '
endif

;--------------
;Optional plot
;--------------
if (keyword_set(NO_PLOT)) then RETURN

;-----------------------------------
;Load color table "Rainbow + white"
;Don't want last color to be white!
;-----------------------------------
loadct, 39, /silent
white = (!d.table_size - 1)
device, decomposed=0
if (keyword_set(PS_File)) then begin
    ;-------------------------------
    ;Will plot to a PostScript file
    ;-------------------------------
    win_system = !d.name
    set_plot, 'PS'
    device, file=PS_file, /color
endif else begin
    ;-------------------------------
    ;Will plot to a graphics window
    ;-------------------------------
    pstr  = New_String(p, FORMAT='(F4.2)')
    nstr  = New_String(n)
    part1 = 'Binomial Cascade (p='
    part2 = pstr + ', n=' + nstr + ')'
    window, 0, xsize=xsize, ysize=ysize, $
            xpos=50, TITLE=(part1 + part2)
    erase, white
endelse

;----------------------------------
;Make sure last color is not white
;----------------------------------
num_colors = !d.table_size
colors     = fix(num_colors *(findgen(n) / float(n)))
colors[n-1L] = colors[n - 2L]

;--------------------------------------
;Initialize vars and use NORMAL coords
;--------------------------------------
dx_min = (2d / (xplot * xsize))
dx     = (xplot / double(n)) > dx_min
ddx    = (dx / 8d) > (dx_min / 2d)

height = yplot * (list / max(list))

;print,'dx   = ',dx
;print,'ddx  = ',ddx
;print,'hmax = ',max(height)

;----------------
;Create the plot
;----------------
if (nlevels le 7) then begin
    for i=0L,(n - 1L) do begin
        x = margin + [ i*dx, (i+1)*dx - ddx, (i+1)*dx - ddx, i*dx]
        y = margin + [ 0, 0, height[i], height[i] ]
        polyfill, x, y, color=colors[i], /NORMAL
    endfor
endif else begin
    xvals = margin + xplot*(dindgen(n)/n)
    for i=0L,(n - 1L) do begin
        x = xvals[i]
        y = margin + height[i]
        plots, [x,x], [margin,y], color=colors[i], /NORMAL
    endfor
endelse

;--------------
;Optional axes
;--------------
;black = 0
;plots, [margin,margin + xplot], [margin,margin], /NORMAL, color=black
;plots, [margin,margin], [margin,margin + yplot], /NORMAL, color=black

;-----------------------------
;Toggle back to window system
;-----------------------------
if (keyword_set(PS_file)) then begin
    device, /close
    set_plot, win_system
endif
print,'Finished.'

end;  Plot_Binomial_Cascade
;**********************************************************************


