
;*****************************************************************
;   bif_diagram.pro    (Created by Rivix, LLC, 1998.)

;-----------------------------------
;   List of procedures in this file
;-----------------------------------
;   Plot_Bifurcation_Diagram

;*****************************************************************
pro Plot_Bifurcation_Diagram, n=n, m=m, AMIN=amin, AMAX=amax, $
                     color=color, XSIZE=XSIZE, YSIZE=YSIZE, $
                     XOFF=XOFF, PS_file=PS_file

;----------------------------------------------------------------
;NOTES : This procedure generates a bifurcation diagram for the
;        logistic map, which is given by f(a,x)= a * x * (1-x).

;        n = number of equally-spaced "a" values
;        m = number of iterations per "a" value (> 200)

;        The idea is to step "a" through its full range by very
;        small increments, and for each "a" to generate many
;        iterates of the map, keeping only the last 200 for the
;        plot.  A matrix "x" is used to store the values, and
;        looks like:

;                 [x1(a1), x2(a1), ... , xm(a1)]
;                 [x1(a2), x2(a2), ... , xm(a2)]
;                     . . . . . . . . . . . 
;                 [x1(an), x2(an), ... , xm(an)]

;        Exponentiating the a-axis of the plot makes it easier
;        to see the "nonchaotic windows." 

;        The "uniq" routine is used to eliminate overplots of
;        the same point, and reduces the size of PostScript
;        files by a factor of around 8.

;        The xsize and ysize keywords set window size.

;        Normal coordinates are used to make the routine
;        device-independent (window or PS file).

;        Set the PS_file keyword to a filename path in
;        single quotes with extension '.ps' in order to
;        save the image to a PostScript file.
;----------------------------------------------------------------

;-----------------
;Keyword defaults
;-----------------
if NOT(keyword_set(XSIZE)) then XSIZE = 500
if NOT(keyword_set(YSIZE)) then YSIZE = 400
if NOT(keyword_set(XOFF))  then XOFF  = 100
if NOT(keyword_set(AMIN))  then amin  = 1.0
if NOT(keyword_set(AMAX))  then amax  = 4.0
if NOT(keyword_set(n)) then n = 200
if NOT(keyword_set(m)) then m = 250
n_str = strtrim(string(n),2)
m_str = strtrim(string(m),2)
print,'n = Number of equally-spaced "a" values = ' + n_str
print,'m = Number of iterations per "a" value  = ' + m_str
print,' '

;----------------
;Initialize vars
;----------------
a = (1.0 + 3*findgen(n)/n)
x = dblarr(m,n)
x[0,*] = 0.56
amin_str = strtrim(string(amin, format='(F4.1)'),2)
amax_str = strtrim(string(amax, format='(F4.1)'),2)
xtitle = 'Exp(a), (' + amin_str + ' < a < ' + amax_str + ')'

;----------------------------------
;Compute the logistic map iterates
;----------------------------------
print,'Working...'
for j=0,(n - 1) do begin
    for k=1,(m-1) do begin
        x[k,j] = (a[j] * x[k-1,j] * (1 - x[k-1,j])) 
    endfor
endfor    

;-----------------------------------
;Load color table "Rainbow + white"
;-----------------------------------
loadct, 39, /silent
black = 0
white = (!d.table_size - 1)
color = black
if (keyword_set(PS_File)) then begin
    ;-------------------------------
    ;Will plot to a PostScript file
    ;-------------------------------
    win_system = !d.name
    set_plot, 'PS'
    device, file=PS_file
endif else begin
    ;-------------------------------
    ;Will plot to a graphics window
    ;-------------------------------
    device, decomposed=0
    window, 0, xsize=xsize, ysize=ysize, $
            xpos=xoff, TITLE='Bifurcation Diagram for Logistic Map'
    erase, white
endelse

print,'Plotting...'
plot,[1],[0],YRANGE=[0,1], XRANGE=[exp(amin),exp(amax)], $    
     xtitle=xtitle, XSTYLE=11, YSTYLE=11, color=color, $
     TITLE='Bifurcation Diagram for the Logistic Map', $
     background=white

;-----------------
;Plot the diagram
;-----------------
for j=0,(n-1) do begin
    y  = x[m-200:m-1,j]
    y  = y[uniq(y,sort(y))]
    ny = n_elements(y)
    if (ny eq 1) then y = [y]
    ea = exp(fltarr(ny) + a[j])
    oplot, ea, y, psym=3, color=color
endfor

;-----------------------------
;Toggle back to window system
;-----------------------------
if (keyword_set(PS_file)) then begin
    device, /close
    set_plot, win_system
endif

print,'Finished with plot.'
print,' '

end;  Plot_Bifurcation_Diagram
;*****************************************************************
