
;*********************************************************************
;   SSTs.pro.    Created by Rivix, LLC, 1998.

;-------------------------------
;   IDL procedures in this file
;-------------------------------
;   Get_SST_Info
;   Plot_SST
;   Cayley_Number
;   Get_TDCNs

;*********************************************************************
pro Get_SST_Info, GENERATORS=GENERATORS, OMAX=OMAX, BN=BN, $
                  PERFECT=PERFECT, HORTON=HORTON, $
                  TOKUNAGA=TOKUNAGA, SHREVE=SHREVE, $
                  LINEAR=LINEAR, MANDELBROT=MANDELBROT, $
                  PEANO=PEANO, RRSST=RRSST, outfile=outfile

;-------------------------------------------------------
;NOTES:  This routine uses recursive formulas for SSTs
;        to compute and display several topological
;        quantities (N, M, C) as a function of Strahler
;        order.
;-------------------------------------------------------

;-----------------
;Keyword defaults
;-----------------
if NOT(keyword_set(OMAX)) then begin
   if (keyword_set(GENERATORS)) then $
        OMAX=(n_elements(GENERATORS)-1) $
   else OMAX=8
endif
if NOT(keyword_set(BN)) then BN=2.0 else BN=float(BN)

;-------------------
;Get the generators
;-------------------
if (keyword_set(GENERATORS)) then T=GENERATORS $
else T=fltarr(OMAX+1)

if (keyword_set(PERFECT)) then begin
    print,'----------------------------'
    print,'Perfect Tree: T(k)=0, all k'
    print,'----------------------------'
endif

if (keyword_set(HORTON)) then begin
    print,'------------------------'
    print,'Hortonian Tree:'
    print,'T(1)=c, T(k)=0, all k>1'
    print,'-------------------------'
    print,'Enter c:'
    read,c  &  T(1)=c
endif

if (keyword_set(TOKUNAGA)) then begin
    print,'------------------------------------'
    print,"Tokunaga's Trees: T(k) = a c^(k-1)"
    print,'------------------------------------'
    print,'Enter a and c:'
    read,a0,c0
    for k=1,OMAX do T(k)=(a0 * c0^(k-1))
    ;***BN=2.0
endif

if (keyword_set(SHREVE)) then begin
    print,'------------------------------------'
    print,'Average Shreve Tree: T(k) = 2^(k-1)'
    print,'------------------------------------'
    for k=1,OMAX do T(k)=2.0^(k-1)
    BN=2.0
endif

if (keyword_set(LINEAR)) then begin
    print,'--------------------------'
    print,'"Linear" Tree: T(k) = a k'
    print,'--------------------------'
    print,'Enter a:'  &  read,a0
    for k=1,OMAX do T(k)=(a0 * k)
endif

if (keyword_set(MANDELBROT)) then begin
    print,'------------------------'
    print,'Mandelbrot-Viscek Tree:'
    print,'T(1) = 0'
    print,'T(k) = 2^(k-2)'
    print,'------------------------'
    for k=2,OMAX do T(k)=2.0^(k-2)
    BN=2.0
endif

if (keyword_set(PEANO)) then begin
    print,'-----------------------'
    print,"Peano's Tree: (BN = 3)"
    print,'T(1) = 0'
    print,'T(k) = 2^(k-1)'
    print,'-----------------------'
    BN=3.0
    for k=2,OMAX do T(k)=2.0^(k-1)
endif

if (keyword_set(RRSST)) then begin
    print,'-------------------------------'
    print,'Recursive-Replacement Tree:'
    print,'T(1) = (n-1)'
    print,'T(k) = (b-1)*(n^2)*(n+1)^(k-2)'
    print,'-------------------------------'
    print,'Enter b and n:'
    read,BN,n  &  BN=float(BN)  &  n=long(n)
    T(1)=(BN-1) * (n-1)
    for k=2,OMAX do T(k)=(BN-1)*(n^2)*(n+1)^(k-2)
endif

;----------------
;Initialize vars
;----------------
N=fltarr(OMAX+1)  &  N(OMAX)=1.0
M=fltarr(OMAX+1)  &  M(1)=1.0
C=fltarr(OMAX+1)  &  C(1)=1.0

;---------------------------
;Compute the stream numbers
;---------------------------
for w=(OMAX-1),1,-1 do begin
    j=(OMAX - w)
    N(w) = (BN*N(w+1) + total(T(1:j)*N(w+1:OMAX)))
endfor
NRatio = N(1:OMAX-1)/N(2:OMAX)
NRatio = [0,NRatio,0]

;-----------------------
;Compute the magnitudes
;-----------------------
for w=2,OMAX do begin
    M2=rotate(M(1:w-1), 2)  ;(reverse)
    M(w) = (BN*M(w-1) + total(T(1:w-1)*M2))
endfor
MRatio = M(2:OMAX)/M(1:OMAX-1)
MRatio = [0,MRatio,0]

;-------------------------
;Compute links per stream
;-------------------------
for w=2,OMAX do begin
    C(w) = (1.0 + total(T(1:w-1)))
endfor
CRatio = C(2:OMAX)/C(1:OMAX-1)
CRatio = [0,CRatio,0]

;-----------------------------
;Print out results in a table
;-----------------------------
f='(I5,F8.1,F10.1,F8.4,F10.1,F8.4,F8.1,F8.4)'
print,' '
;00001000000010000000001000000010000000001000000010000000100000001
print,'---------------------------------' + $
         '---------------------------------'
print,'  Order    T         N   NRatio        ' + $
      'M   MRatio      C   CRatio'
print,'---------------------------------' + $
         '---------------------------------'

for w=1,OMAX do begin
    INFOSTR = string(w,T(w),N(w),NRatio(w),M(w),MRatio(w), $
                     C(w),CRatio(w), format=f, /PRINT)
    print,INFOSTR
endfor
print,' '

;---------------------------------
;Option to save results in a file
;---------------------------------
if (keyword_set(outfile)) then begin 
    openw,U,outfile,GET_LUN
    printf,U,' '
    printf,U,'a = ',a0,', c = ',c0
    printf,U,' '
    printf,U,'---------------------------------' + $
             '---------------------------------'
    printf,U,'  Order    T         N   NRatio        ' + $
             'M   MRatio      C   CRatio'
    printf,U,'---------------------------------' + $
             '---------------------------------'

    for w=1,OMAX do begin
        printf,U,w,T(w),N(w),NRatio(w),M(w),MRatio(w), $
               C(w),CRatio(w), format=f
    endfor
    printf,U,' '
    free_lun,U
endif

end;  Get_SST_Info
;*****************************************************************
pro Plot_SST, order=order, x0=x0, y0=y0, angle=angle, $
              L=L, NL=NL, mix=mix, sides=sides, $
              NOT_ROOT=NOT_ROOT, color=color, thick=thick, $ 
              PERFECT=PERFECT, HORTON=HORTON, $
              SHREVE=SHREVE, MANDELBROT=MANDELBROT, $
              XSIZE=xsize, YSIZE=ysize, XOFF=xoff, $
              PS_file=PS_file

;------------------------------------------------------------
;NOTES:  Since this is a recursive routine, the NOT_ROOT
;        keyword is used to distinguish the initial call
;        from the recursive calls.  For example, we only
;        need to open a graphics window on initial call.

;        Note:  A treetype keyword can only be set in
;        the initial call.

;        x0    = x-coordinate of tree root
;        y0    = y-coordinate of tree root
;        order = w = Strahler order of the tree
;        angle = angle trunk stream makes with horizontal
;        L(w)  = length of an order w stream
;        NL(w) = number of links in an order w stream
;        mix   = interspersal array for the highest order
;                stream
;                (e.g. (T3,T2,T3,T1,T3,T2,T3)
;                    -> (3, 2, 3, 1, 3, 2, 3). )
;        sides = left/right side array (1's and -1's)

;        Normal coordinates are used to make the routine
;        device-independent (window or PS file).

;        Set the PS_file keyword to a filename path in
;        single quotes with extension '.ps' in order to
;        save the image to a PostScript file.
;-------------------------------------------------------------

;-----------------
;Keyword defaults
;-----------------
if NOT(keyword_set(order)) then order=5
order = ((order > 1) < 8)
if NOT(keyword_set(x0)) then x0=0.5d
if NOT(keyword_set(y0)) then y0=0.05d
if NOT(keyword_set(L)) then begin
    a = (0.5d / 2.0d^(order-1) )
    L = [0,a,2*a,4*a,8*a,16*a,32*a,64*a,128*a,256*a]
endif
if NOT(keyword_set(angle)) then angle=(!DPI/2)
if NOT(keyword_set(color)) then color=RT_Color_Index('black')
if NOT(keyword_set(thick)) then thick=1
if NOT(keyword_set(XSIZE)) then xsize=500
if NOT(keyword_set(YSIZE)) then ysize=450
if NOT(keyword_set(XOFF))  then xoff=200

NOT_ROOT   = keyword_set(NOT_ROOT)
PERFECT    = keyword_set(PERFECT)
HORTON     = keyword_set(HORTON)
SHREVE     = keyword_set(SHREVE)
MANDELBROT = keyword_set(MANDELBROT)

;-----------------------------------
;Default to the Average Shreve Tree
;-----------------------------------
KEY_SUM = (PERFECT + HORTON + SHREVE + MANDELBROT)
if (KEY_SUM eq 0) AND NOT(NOT_ROOT) then SHREVE = 1b

;---------------------------
;Perfect Tree  T(k)=0, k>0
;---------------------------
if (PERFECT) then begin
    treename = 'Perfect Tree'
    NL=[0,1,1,1,1,1,1,1,1,1]
    mix=[0]
    sides=[0]
endif

;----------------------------------
;Horton Tree.  T(1)=1, T(k)=0, k>1
;----------------------------------
if (HORTON) then begin
    treename = 'Hortonian (or cyclic) Tree'
    NL=[0,2,2,2,2,2,2,2,2,2]
    mix=[0,1]
    sides=[0,-1]
endif

;----------------------------------
;Avg. Shreve Tree.  T[k] = 2^(k-1)
;----------------------------------
if (SHREVE) then begin
    treename = 'Average Shreve Tree'
    NL = [0, 2^indgen(order + 1)]
    ;-----------------------------
    ;Build the interspersal array
    ;-----------------------------
    mix=[order,order-1,order]
    for j=(order-2),1,-1 do mix=[mix,j,mix]
    mix=[0,mix]

    ;------------------------
    ;Build the "sides" array
    ;------------------------ 
    sides=[1,-1,1]
    for j=(order-2),1,-1 do sides=(-1 * [sides,1,sides])
    sides=[0,sides]
endif

;-------------------------------------------
;Mandelbrot Tree. T(1)=0, T(k)=2^(k-2), k>1
;-------------------------------------------
if (MANDELBROT) then begin
    treename = 'Mandelbrot-Vicsek Tree'
    NL=[0,1,1,2,4,8,16,32,64,128,256]    
    mix=[0,4,3,4,2,4,3,4]          ;(* EXPAND THESE *)
    sides=[0,-1,1,1,-1,1,1,-1]
endif

;-------------------------------------
;Prepare to plot the self-similar tree
;-------------------------------------
if NOT(NOT_ROOT) then begin
    ;-----------------------------------
    ;Load color table "Rainbow + white"
    ;-----------------------------------
    loadct, 39, /silent
    black = 0
    white = (!d.table_size - 1)
    color = black
    device, decomposed=0
    if (keyword_set(PS_File)) then begin
        ;-------------------------------
        ;Will plot to a PostScript file
        ;-------------------------------
        win_system = !d.name
        set_plot, 'PS'
        device, file=PS_file
    endif else begin
        ;-------------------------------
        ;Will plot to a graphics window
        ;-------------------------------
        window, 0, xsize=xsize, ysize=ysize, xpos=xoff, $
                TITLE=treename
        erase, white
    endelse

    ordstr = strtrim(string(order),2) + ')'
    print,'Plotting ' + treename + '...' 
    print,'  (Order = ' + ordstr
endif

;---------------------------------
;Draw the stream of order "order"
;---------------------------------
x1 = x0 + (L[order] * cos(angle))
y1 = y0 + (L[order] * sin(angle))
plots,[x0,x1],[y0,y1], color=color, thick=thick, /normal

if (order gt 1) then begin
    ;--------------------------------
    ;Recursive call to draw the left
    ;upstream tributary network
    ;--------------------------------
    leftangle = angle + (!PI/4.0)
    Plot_SST, order=(order-1), x0=x1, y0=y1, /NOT_ROOT, $
              angle=leftangle, L=L, NL=NL, mix=mix, $
              sides=sides, color=color, thick=thick

    ;---------------------------------
    ;Recursive call to draw the right
    ;upstream tributary network
    ;---------------------------------
    rightangle = angle - (!PI/4.0)
    Plot_SST, order=(order-1), x0=x1, y0=y1, /NOT_ROOT, $
              angle=rightangle, L=L, NL=NL, mix=mix, $
              sides=sides, color=color, thick=thick

    ;---------------------------
    ;Recursive call to draw the
    ;side tributary networks
    ;---------------------------
    s = where(mix lt order, n)
    n = (n-1)                       ;(skip over the 0.)
    localmix   = mix[s]
    localsides = sides[s]  
    for trib=1,n do begin   
        k   = localmix[trib]
        sgn = localsides[trib]

        x2 = x0 + (L[order] / NL[order] * trib * cos(angle) )
        y2 = y0 + (L[order] / NL[order] * trib * sin(angle) )

        sideangle = angle + (sgn * !PI/2.5)
        Plot_SST, order=(order-k), x0=x2, y0=y2, /NOT_ROOT, $
                  angle=sideangle, L=L, NL=NL, mix=mix, $
                  sides=sides, color=color, thick=thick
    endfor
 
endif  ;**** (IF (order gt 1)) ****

if NOT(NOT_ROOT) then begin
   ;-----------------------------
   ;Toggle back to window system
   ;-----------------------------
   if (keyword_set(PS_file)) then begin
       device, /close
       set_plot, win_system
   endif
   print,'Finished.'
endif

end;  Plot_SST 
;*********************************************************************
function Cayley_Number, m

;----------------------------------------------------
;NOTES:  This routine computes the so-called Cayley
;        (or Catalan) numbers, which are given by:
;           N(m) = ((2m - 1) choose m)/(2m - 1)
;        using a robust algorithm, which can compute
;        them for (m le 20).
;           N(20) = 1767263488L
;----------------------------------------------------
num = 1.0
for k=2L,(m-1L) do num = (num * float(2*m - k)/float(k))
 
RETURN, long(num)
end;    Cayley_Number
;*********************************************************************
pro Get_TDCNs, m, list, outfile=outfile, REPORT=REPORT
 
;-------------------------------------------------------------
;NOTES:  This procedure generates a list of all possible
;        topologically distinct channel networks (TDCNs) of
;        a given magnitude, m.
 
;        Network trees are listed as sequences of 0's and 1's
;        according to Shreve's "left-first" encoding scheme
;        for binary trees.  Interior links are 1's and
;        exterior (or leaf) links are 0's.  A Rubik's Snake
;        can be used to visualize the random walk sequences.

;        This version uses a more elegant recursive algorithm
;        which never double counts and produces the list in
;        sorted order to begin with.

;        This routine uses the Cayley_Number function above,
;        and should in principle work for m up to 20,
;        given a computer with a huge amount of RAM.  A more
;        practical limit is (m eq 15).

;        m    = Shreve magnitude (number of leaves)
;        num0 = number of 0's in the sequence
;        num1 = number of 1's in the sequence
;        SP   = stack pointer
;-------------------------------------------------------------
 
;--------------------
;Check the magnitude
;--------------------
if (n_elements(m) eq 0) then begin
    print,'Get_TDCNs called with no arguments.'
    RETURN
endif
if (m le 1) then begin
    print,'Magnitude must be greater than 1.'
    RETURN
endif

;------------------------------
;How many TDCNS will there be?
;------------------------------
c = Cayley_Number(m)

;--------------
;Print message
;--------------
MSTR = strtrim(string(m),2)
CSTR = ' = ' + strtrim(string(c),2)
print,' '
print,'Number of TDCNS for magnitude ' + MSTR + CSTR 
print,' '

;----------------
;Initialize vars
;----------------
print,'Initializing arrays...'
numlinks  = (2*m - 1)
stack     = strarr(m)
stacknum0 = bytarr(m)
stacknum1 = bytarr(m)
list      = strarr(c)
tree='1'  &  num0=0  &  num1=1
SP=1L     &  n=0L

;---------------------------------
;Generate TDCN list "recursively"
;---------------------------------
print,'Generating list...'
repeat begin
    repeat begin
        ;------------------------------
        ;Add as many zeros as possible
        ;------------------------------
        repeat begin
            if (num1 lt (m-1)) then begin
               ;------------------------
               ;Put "tree + 1" on stack
               ;------------------------
               stack[SP] = tree + '1'
               stacknum0[SP] = num0
               stacknum1[SP] = (num1 + 1)
               SP = (SP + 1L)
            endif 

            tree = tree + '0'
            num0 = num0 + 1 
        endrep until ((num0 eq num1) OR (num0 eq (m-1)))

        ;-----------------
        ;Try adding a one
        ;-----------------
        if (num1 lt (m-1)) then begin
            tree = tree + '1'
            num1 = num1 + 1 
        endif

        ;--------------
        ;Update length
        ;--------------
        length = (num0 + num1)

    endrep until (length eq (numlinks-1))

    ;----------------
    ;Save the "tree"
    ;----------------
    list[n] = tree + '0'
    n = (n + 1L)

    ;---------------------
    ;Read next from stack
    ;---------------------
    SP   = (SP - 1L)
    tree = stack[SP]
    num0 = stacknum0[SP]
    num1 = stacknum1[SP]

endrep until (SP eq 0L)

list = transpose(temporary(list))

;---------------------
;Option to print list
;---------------------
if (keyword_set(REPORT)) then begin
    print,list
    print,' '
endif
print,'Finished.'
print,' '
 
;--------------------------------
;Option to save TDCNs in outfile
;--------------------------------
if (keyword_set(outfile)) then begin
    openw, UNIT, outfile, /get_lun
    printf, UNIT, 'Number of TDCNs for magnitude ' + MSTR + CSTR
    printf, UNIT, ' ' 
    printf, UNIT, list
    free_lun, UNIT
endif

end;  Get_TDCNs
;*****************************************************************
